package org.msh.tb.bd.tbforms.input;

import org.apache.commons.beanutils.PropertyUtils;
import org.jboss.seam.Component;
import org.jboss.seam.annotations.In;
import org.msh.etbm.commons.transactionlog.DetailXMLWriter;
import org.msh.tb.bd.Quarter;
import org.msh.tb.entities.*;
import org.msh.tb.entities.enums.RoleAction;
import org.msh.tb.entities.enums.TbFormField;
import org.msh.tb.indicators.core.IndicatorFilters;
import org.msh.tb.login.UserSession;

import javax.persistence.EntityManager;
import java.util.Date;
import java.util.List;

/**
 * Created by Mauricio on 06/03/2017.
 * Abstract class used by TB Form Home instances
 * It abstracts the initialize and save methods, needing only to specify some data on its child.
 */
public abstract class InputFormHome {

    @In
    EntityManager entityManager;

    /**
     * Component that stores selected unit and quarter
     */
    IndicatorFilters indicatorFilter;

    /**
     * @return an array containing the fields of the report in question
     */
    protected abstract TbFormField[] getFields();

    /**
     * @return the report name to be set on transaction log
     */
    protected abstract String getReportName();

    /**
     * @return the DTO class that stores the values
     */
    public abstract Object getData();

    /**
     * Initialized indicatorFilters state according to user workspace logged in
     */
    public void initializeUnitView() {

        // Set users unit as selected unit
        UserWorkspace uw = UserSession.getUserWorkspace();
        getIndicatorFilters().getTbunitselection().getAuselection().setSelectedUnit(uw.getTbunit().getAdminUnit());
        getIndicatorFilters().getTbunitselection().setSelected(uw.getTbunit());

        // If quarter is empty set current quarter
        Quarter quarter = getIndicatorFilters().getQuarter();
        if (quarter == null || quarter.getYear() == 0 || quarter.getQuarter() == null) {
            quarter = Quarter.getCurrentQuarter().getPreviousQuarter();
            getIndicatorFilters().setQuarterYear(quarter.getYear());
            getIndicatorFilters().setQuarterMonths(quarter.getQuarter());
        }
    }

    /**
     * Initialized DTO setting 0 to all fields.
     * After that, searched on db the current values for each field. If exists, set it on DTO.
     */
    public void initialize() {

        if (UserSession.isTbFormUnitView()) {
            initializeUnitView();
        } else {
            // avoid lazy init
            getIndicatorFilters().getTbunitselection().setSelected(entityManager.find(Tbunit.class, getSelectedTbunit().getId()));
        }

        if (getSelectedTbunit() == null) {
            throw new RuntimeException("Tbunit can't be null.");
        }

        Quarter quarter = getQuarter();
        if (quarter == null || quarter.getYear() == 0 || quarter.getQuarter() == null) {
            throw new RuntimeException("Quarter can't be empty.");
        }

        List<TbFormValue> values = entityManager.createQuery("from TbFormValue v " +
                "where v.tbunit.id = :tbunitId and v.referenceDate = :refDate and v.field in " + getFieldsIds())
                .setParameter("tbunitId", getSelectedTbunit().getId())
                .setParameter("refDate", getQuarter().getIniDate())
                .getResultList();

        try {

            // set all fields as zero
            for (TbFormField field : getFields()) {
                PropertyUtils.setProperty(getData(), field.getDtoFieldName(), new Integer(0));
            }

            // fill in existing values
            for (TbFormValue formValue : values) {
                PropertyUtils.setProperty(getData(), formValue.getField().getDtoFieldName(), formValue.getValue());
            }

        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Deletes all current values and then insert the new values on db
     * @return success if success
     */
    public String save() {
        // deletes all current values on db the refers to the selected form, unit and quarter
        entityManager.createQuery("delete from TbFormValue v " +
                "where v.tbunit.id = :tbunitId and v.referenceDate = :refDate and v.field in  " + getFieldsIds())
                .setParameter("tbunitId", getSelectedTbunit().getId())
                .setParameter("refDate", getQuarter().getIniDate())
                .executeUpdate();

        // insert the new values on database
        for (TbFormField field : getFields()) {
            Integer value;

            try {
                value = (Integer) PropertyUtils.getProperty(getData(), field.getDtoFieldName());
            } catch (Exception e) {
                throw new RuntimeException(e);
            }

            // inserts if it is not null and bigger than zero
            if (value != null && value > 0) {
                TbFormValue tbFormValue = new TbFormValue();
                tbFormValue.setField(field);
                tbFormValue.setReferenceDate(getQuarter().getIniDate());
                tbFormValue.setTbunit(getSelectedTbunit());
                tbFormValue.setValue(value);

                entityManager.persist(tbFormValue);
            }
        }

        // save a transaction log
        saveTransactionLog();

        return "success_" + (UserSession.isTbFormUnitView() ? "unit" : "manag");
    }

    /**
     * Saves a log for the saving operation
     */
    private void saveTransactionLog() {
        // get logged user
        UserWorkspace userWorkspace = (UserWorkspace) Component.getInstance("userWorkspace", true);

        UserLog userLog = entityManager.find(UserLog.class, userWorkspace.getUser().getId());
        WorkspaceLog workspaceLog = entityManager.find(WorkspaceLog.class, userWorkspace.getWorkspace().getId());
        UserRole role = (UserRole) entityManager.createQuery("from UserRole where name = :name")
                .setParameter("name", "TB_FORM_INPUT")
                .getResultList().get(0);

        DetailXMLWriter detailWriter = new DetailXMLWriter();

        detailWriter.addTable();
        detailWriter.addTableRow("Tb Unit", getSelectedTbunit().getName() + " - " + getSelectedTbunit().getAdminUnit().getFullDisplayName2());
        detailWriter.addTableRow("Quarter", getQuarter().getDHIS2QuarterCode());
        detailWriter.addTableRow("TB Form:", getReportName());

        // write each field
        for (TbFormField field : getFields()) {
            try {
                Integer value = (Integer) PropertyUtils.getProperty(getData(), field.getDtoFieldName());
                detailWriter.addTableRow(field.getKey(), value);
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        }

        // save the transaction log
        TransactionLog log = new TransactionLog();
        log.setAction(RoleAction.EDIT);
        log.setRole(role);
        log.setTransactionDate(new Date());
        log.setUser(userLog);
        log.setWorkspace(workspaceLog);
        log.setAdminUnit(getSelectedTbunit().getAdminUnit());
        log.setUnit(getSelectedTbunit());
        log.setComments(detailWriter.asXML());

        entityManager.persist(log);
    }

    /**
     * @return the list of ordinal  of each field on getFields result. Used to mount the queries on this service.
     */
    private String getFieldsIds(){
        String ids = "(";

        for (TbFormField field : getFields()) {
            ids = ids.concat(field.ordinal() + ",");
        }

        ids = ids.substring(0, ids.length()-1);

        return ids + ")";
    }

    public Quarter getQuarter() {
        return getIndicatorFilters().getQuarter();
    }

    public Tbunit getSelectedTbunit() {
        return getIndicatorFilters().getTbunitselection().getSelected();
    }

    /**
     * Return the filters selected by the user
     * @return instance of the {@link IndicatorFilters IndicatorFilters} class
     */
    protected IndicatorFilters getIndicatorFilters() {
        if (indicatorFilter == null)
            indicatorFilter = (IndicatorFilters) Component.getInstance("indicatorFilters");
        return indicatorFilter;
    }
}
